/*
 * Decompiled with CFR 0.152.
 */
package net.pcal.fastback.repo;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.text.ParseException;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import net.pcal.fastback.config.FastbackConfigKey;
import net.pcal.fastback.config.GitConfig;
import net.pcal.fastback.config.OtherConfigKey;
import net.pcal.fastback.logging.SystemLogger;
import net.pcal.fastback.logging.UserLogger;
import net.pcal.fastback.logging.UserMessage;
import net.pcal.fastback.mod.Mod;
import net.pcal.fastback.repo.BranchUtils;
import net.pcal.fastback.repo.CommitUtils;
import net.pcal.fastback.repo.JGitSupplier;
import net.pcal.fastback.repo.PruneUtils;
import net.pcal.fastback.repo.PushUtils;
import net.pcal.fastback.repo.ReclamationUtils;
import net.pcal.fastback.repo.Repo;
import net.pcal.fastback.repo.RestoreUtils;
import net.pcal.fastback.repo.SnapshotId;
import net.pcal.fastback.repo.SnapshotIdUtils;
import net.pcal.fastback.repo.WorldId;
import net.pcal.fastback.repo.WorldIdUtils;
import net.pcal.fastback.utils.EnvironmentUtils;
import net.pcal.fastback.utils.ProcessException;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.errors.NoWorkTreeException;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.util.FileUtils;

class RepoImpl
implements Repo {
    static final String FASTBACK_DIR = ".fastback";
    private final Git jgit;
    private GitConfig config;
    private WorldIdUtils.WorldIdInfo worldIdInfo;

    RepoImpl(Git jgit) {
        this.jgit = Objects.requireNonNull(jgit);
    }

    @Override
    public void doCommitAndPush(UserLogger ulog) {
        SnapshotId newSid;
        if (!this.doNativeCheck(ulog)) {
            return;
        }
        this.checkIndexLock(ulog);
        this.broadcastBackupNotice();
        long start = System.currentTimeMillis();
        try {
            newSid = CommitUtils.doCommitSnapshot(this, ulog);
        }
        catch (IOException | ProcessException | GitAPIException e) {
            SystemLogger.syslog().error(e);
            ulog.message(UserMessage.styledLocalized("fastback.chat.commit-failed", UserMessage.UserMessageStyle.ERROR, new Object[0]));
            return;
        }
        try {
            PushUtils.doPush(newSid, this, ulog);
        }
        catch (IOException | ProcessException e) {
            ulog.message(UserMessage.styledLocalized("fastback.chat.push-failed", UserMessage.UserMessageStyle.ERROR, new Object[0]));
            SystemLogger.syslog().error(e);
            return;
        }
        ulog.message(UserMessage.localized("fastback.chat.backup-complete-elapsed", RepoImpl.getDuration(start)));
    }

    @Override
    public void doCommitSnapshot(UserLogger ulog) {
        if (!this.doNativeCheck(ulog)) {
            return;
        }
        this.checkIndexLock(ulog);
        this.broadcastBackupNotice();
        long start = System.currentTimeMillis();
        try {
            SnapshotId newSid = CommitUtils.doCommitSnapshot(this, ulog);
        }
        catch (IOException | ProcessException | GitAPIException e) {
            ulog.message(UserMessage.styledLocalized("fastback.chat.commit-failed", UserMessage.UserMessageStyle.ERROR, new Object[0]));
            SystemLogger.syslog().error(e);
            return;
        }
        ulog.message(UserMessage.localized("fastback.chat.backup-complete-elapsed", RepoImpl.getDuration(start)));
    }

    @Override
    public void doPushSnapshot(SnapshotId sid, UserLogger ulog) {
        if (!this.getConfig().isSet(OtherConfigKey.REMOTE_PUSH_URL)) {
            ulog.message(UserMessage.styledLocalized("No remote is configured.  Run set-remote <url>", UserMessage.UserMessageStyle.ERROR, new Object[0]));
            return;
        }
        if (!this.doNativeCheck(ulog)) {
            return;
        }
        long start = System.currentTimeMillis();
        try {
            PushUtils.doPush(sid, this, ulog);
        }
        catch (IOException | ProcessException e) {
            ulog.message(UserMessage.styledLocalized("fastback.chat.commit-failed", UserMessage.UserMessageStyle.ERROR, new Object[0]));
            SystemLogger.syslog().error(e);
            return;
        }
        ulog.message(UserMessage.localized("Successfully pushed " + sid.getShortName() + ".  Time elapsed: " + RepoImpl.getDuration(start), new Object[0]));
    }

    @Override
    public Collection<SnapshotId> doLocalPrune(UserLogger ulog) throws IOException {
        return PruneUtils.doLocalPrune(this, ulog);
    }

    @Override
    public Collection<SnapshotId> doRemotePrune(UserLogger ulog) throws IOException {
        return PruneUtils.doRemotePrune(this, ulog);
    }

    @Override
    public void doGc(UserLogger ulog) {
        if (!this.doNativeCheck(ulog)) {
            return;
        }
        try {
            ReclamationUtils.doReclamation(this, ulog);
        }
        catch (ProcessException | GitAPIException e) {
            ulog.message(UserMessage.styledLocalized("Command failed.  Check log for details.", UserMessage.UserMessageStyle.ERROR, new Object[0]));
            SystemLogger.syslog().error(e);
        }
    }

    @Override
    public void doRestoreLocalSnapshot(String snapshotName, UserLogger ulog) {
        RestoreUtils.doRestoreLocalSnapshot(snapshotName, this, ulog);
    }

    @Override
    public void doRestoreRemoteSnapshot(String snapshotName, UserLogger ulog) {
        RestoreUtils.doRestoreRemoteSnapshot(snapshotName, this, ulog);
    }

    @Override
    public WorldId getWorldId() throws IOException {
        return WorldIdUtils.getWorldIdInfo(this.getWorkTree().toPath()).wid();
    }

    @Override
    public Set<SnapshotId> getLocalSnapshots() throws IOException {
        JGitSupplier<Collection<Ref>> refProvider = () -> {
            try {
                return this.jgit.branchList().call();
            }
            catch (GitAPIException e) {
                throw new IOException(e);
            }
        };
        try {
            return BranchUtils.listSnapshots(this, refProvider);
        }
        catch (GitAPIException e) {
            throw new IOException(e);
        }
    }

    @Override
    public Set<SnapshotId> getRemoteSnapshots() throws IOException {
        GitConfig conf = GitConfig.load(this.jgit);
        String remoteName = conf.getString(FastbackConfigKey.REMOTE_NAME);
        JGitSupplier<Collection<Ref>> refProvider = () -> {
            try {
                return this.jgit.lsRemote().setRemote(remoteName).setHeads(true).call();
            }
            catch (GitAPIException e) {
                throw new IOException(e);
            }
        };
        try {
            return BranchUtils.listSnapshots(this, refProvider);
        }
        catch (GitAPIException e) {
            throw new IOException(e);
        }
    }

    @Override
    public GitConfig getConfig() {
        if (this.config == null) {
            this.config = GitConfig.load(this.jgit);
        }
        return this.config;
    }

    @Override
    public File getDirectory() throws NoWorkTreeException {
        return this.jgit.getRepository().getDirectory();
    }

    @Override
    public File getWorkTree() throws NoWorkTreeException {
        return this.jgit.getRepository().getWorkTree();
    }

    @Override
    public void deleteRemoteBranch(String remoteBranchName) throws IOException {
        PruneUtils.deleteRemoteBranch(this, remoteBranchName);
    }

    @Override
    public void deleteLocalBranches(List<String> branchesToDelete) throws IOException {
        PruneUtils.deleteLocalBranches(this, branchesToDelete);
    }

    @Override
    public void close() {
        this.getJGit().close();
    }

    @Override
    public SnapshotId createSnapshotId(String shortName) throws IOException, ParseException {
        return this.getWorldIdInfo().sidCodec().create(this.getWorldId(), shortName);
    }

    SnapshotIdUtils.SnapshotIdCodec getSidCodec() throws IOException {
        return this.getWorldIdInfo().sidCodec();
    }

    Git getJGit() {
        return this.jgit;
    }

    Path getDotFasbackDir() {
        return this.getWorkTree().toPath().resolve(FASTBACK_DIR);
    }

    private WorldIdUtils.WorldIdInfo getWorldIdInfo() throws IOException {
        if (this.worldIdInfo == null) {
            this.worldIdInfo = WorldIdUtils.getWorldIdInfo(this.getWorkTree().toPath());
        }
        return this.worldIdInfo;
    }

    private void checkIndexLock(UserLogger ulog) {
        File lockFile = this.getWorkTree().toPath().resolve(".git/index.lock").toFile();
        if (lockFile.exists()) {
            ulog.message(UserMessage.styledRaw(lockFile.getAbsolutePath() + "exists", UserMessage.UserMessageStyle.WARNING));
            if (this.getConfig().getBoolean(FastbackConfigKey.IS_LOCK_CLEANUP_ENABLED)) {
                ulog.message(UserMessage.styledRaw("lock-cleanup-enabled = true, attempting to delete index.lock", UserMessage.UserMessageStyle.WARNING));
                try {
                    FileUtils.delete((File)lockFile, (int)2);
                }
                catch (IOException e) {
                    SystemLogger.syslog().debug(e);
                }
                if (lockFile.exists()) {
                    ulog.message(UserMessage.styledRaw("Cleanup failed.  Your backup will probably not succeed.", UserMessage.UserMessageStyle.ERROR));
                } else {
                    ulog.message(UserMessage.styledRaw("Cleanup succeeded, proceeding with backup.  But if you see this message again, you should check your system to see if some other git process is accessing your backup.", UserMessage.UserMessageStyle.WARNING));
                }
            } else {
                ulog.message(UserMessage.styledRaw("Please check to see if other processes are using this git repo.  If you're sure they aren't, you can enable automatic index.lock cleanup by typing '/set lock-cleanup enabled'", UserMessage.UserMessageStyle.WARNING));
                ulog.message(UserMessage.styledRaw("Proceeding with backup but it will probably not succeed.", UserMessage.UserMessageStyle.WARNING));
            }
        }
    }

    private static String getDuration(long since) {
        Duration d = Duration.of(System.currentTimeMillis() - since, ChronoUnit.MILLIS);
        long seconds = d.getSeconds();
        if (seconds < 60L) {
            return String.format("%ds", seconds == 0L ? 1L : seconds);
        }
        return String.format("%dm %ds", seconds / 60L, seconds % 60L);
    }

    private void broadcastBackupNotice() {
        if (!this.getConfig().getBoolean(FastbackConfigKey.BROADCAST_ENABLED)) {
            return;
        }
        String configuredMessage = this.getConfig().getString(FastbackConfigKey.BROADCAST_MESSAGE);
        UserMessage m = configuredMessage != null ? UserMessage.styledRaw(configuredMessage, UserMessage.UserMessageStyle.BROADCAST) : UserMessage.styledLocalized("fastback.broadcast.message", UserMessage.UserMessageStyle.BROADCAST, new Object[0]);
        Mod.mod().sendBroadcast(m);
    }

    private boolean doNativeCheck(UserLogger ulog) {
        GitConfig config = this.getConfig();
        if (config.getBoolean(FastbackConfigKey.IS_NATIVE_GIT_ENABLED) && !EnvironmentUtils.isNativeGitInstalled()) {
            ulog.message(UserMessage.styledRaw("Unable to backup: native mode enabled but git is not installed.", UserMessage.UserMessageStyle.ERROR));
            return false;
        }
        return true;
    }
}

